define([
    'underscore',
    'backbone',
    'modules/new-appointment-request/resources/common/model',
    'modules/new-appointment-request/views/common/method/layout-view',
    'modules/new-appointment-request/tests/helpers/facility-helper',
    'modules/new-appointment-request/resources/direct/clinics/collection',
], function(_, Backbone, BaseModel, View, helpers, ClinicsCollection) {
    'use strict';


    describe('Scheduling Method Layout View', function() {

        var region;
        var view;
        var model;


        /**
         * Helper function to show the view
         * @param {Boolean} [isRequest=false]
         * @param {Boolean} [isDirect=false]
         * @param {Boolean} [isPrimaryCare=false]
         * @return {Backbone.View<Backbone.Model>}
         */
        function showView(isRequest, isDirect, isPrimaryCare) {
            var options;
            var layoutView;

            var _isRequest = isRequest || false;
            var _isDirect = isDirect || false;
            var _isPrimaryCare = isPrimaryCare || false;

            var typeOfCareId = _isPrimaryCare ? '323' : '0';
            var typeOfCare = new Backbone.Model({
                'id': typeOfCareId,
                'name': 'whatever',
                'objectType': 'TypesOfCare',
                'link': [ ],
            });

            model.isPrimaryCare = function() {
                return _isPrimaryCare;
            };

            model.set('typeOfCare', typeOfCare);
            model.set('facility', new Backbone.Model({institutionCode: '0'}));
            model.set('unfilteredClinics', new ClinicsCollection());

            options = {
                isRequestSupported: _isRequest,
                isDirectSchedulingSupported: _isDirect,
                requestHistory: null,
                model: model,
            };

            layoutView = new View(options);
            layoutView.filteredClinics = new ClinicsCollection();
            region.show(layoutView);
            return region.currentView;
        }


        beforeEach(function() {
            region = helpers.setupTestingRegion();
            model = new BaseModel();
        });

        afterEach(helpers.resetTestingRegion);


        /**
         * As a user
         * I want to be able to select the specific scheduling method
         * So that I can choose to submit a request or book the appointment directly
         */
        describe('VAR-7912 Specification', function() {


            /**
             * 1.  Label: Step 3: Select Scheduling Method
             */
            describe('1.0', function() {
                it('generates the correct title label', function() {
                    var heading;
                    var text;

                    view = showView();
                    heading = view.$('.form-section-heading')[0];
                    text = heading.innerText.trim();

                    expect(text).toBe('Step 4: Scheduling Method');
                });
            });


            /**
             * 2.  Instructional text: The location you selected supports the following scheduling methods:
             */
            describe('2.0', function() {

                it('generates the correct instructions', function() {
                    var expected;
                    var instructions;
                    var text;

                    view = showView();
                    expected = 'The location you selected supports the following scheduling methods:';
                    instructions = view.$('.step-instructions')[0];
                    text = instructions.innerText.trim();

                    expect(text).toBe(expected);
                });
            });


            /**
             * 3. Label: Select a scheduling method:
             */
            describe('3.0', function() {
                it('generates the correct dropdown label', function() {
                    var label;
                    var text;

                    view = showView();
                    label = view.$('.label-text')[0];
                    text = label.innerHTML.trim();

                    expect(text).toBe('Select a scheduling method:');
                });
            });
        });

        describe('Coverage and Additional Tests', function() {

            beforeEach(function() {
                helpers.collectionFetchSpy();
            });

            it('sets isDirectHistoryNeeded to correctly', function() {
                var _view = showView(false, true, false);
                expect(_view.options.isDirectHistoryNeeded).toBeTruthy();
            });

            it('sets isLimitNeeded to correctly', function() {
                var _view = showView(true);
                expect(_view.options.isLimitNeeded).toBeTruthy();
                expect(_view.options.isRequestHistoryNeeded).toBeTruthy();
            });

            it('will not show unless all needed data is already fetched', function() {
                var _view = showView(true, true, true);
                var _region;
                var sectionView;

                _view.showSectionView();
                _region = _view.getRegion('section');
                sectionView = _region.currentView;

                expect(sectionView).toBeUndefined();
            });

            describe('on fetch success', function() {
                it('sets the model and calls the show function', function() {
                    var _view = showView(false, true, true);
                    var fakePactCollection = new Backbone.Model({'test': 'data'});
                    var _region;
                    var SectionView;
                    var checkModel;

                    fakePactCollection.hasPacTeamAtFacility = _.noop;
                    _view._onFetchSuccess('pacTeam', fakePactCollection);
                    _view._onFetchSuccess('clinics', new Backbone.Collection());
                    _region = _view.getRegion('section');
                    SectionView = _region.currentView;
                    checkModel = SectionView.getOption('pacTeam');

                    expect(checkModel.get('test')).toBe('data');
                });
            });
        });
    });
});
